local assets = {
    Asset("ANIM", "anim/ghost_orchid.zip"),
    Asset("SOUND", "sound/common.fsb"),
}

local prefabs = {
    "ghost_orchid_seed",
    "resurrectionstone_fx",
    "ghost_transform_override",
}

-- 阶段常量
local STAGE = {
    ONE = 1,
    TRANSITION = 2,
    TWO = 3,
    THREE = 4
}

-- 阶段1：喂食怪物肉
local function OnFeedMonsterMeat(inst, giver, item)
    if item.prefab == "monstermeat" and inst.stage == STAGE.ONE then
        inst.SoundEmitter:PlaySound("dontstarve/common/ghost_spawn")
        inst.AnimState:PlayAnimation("ghost_orchid_stage_1_to_2")
        
        inst:DoTaskInTime(1, function()
            inst.stage = STAGE.TRANSITION
            inst:PushEvent("ghost_orchid_stage_changed", {stage = inst.stage})
            inst.AnimState:PlayAnimation("ghost_orchid_transition_idle", true)
            
            -- 更新名称
            if inst.components.named then
                inst.components.named:SetName(STRINGS.NAMES.GHOST_ORCHID_TRANSITION)
            end
            
            -- 设置计时器，5天后进入阶段3
            inst.stage_task = inst:DoTaskInTime(0.005 * TUNING.TOTAL_DAY_TIME, function()
                inst.AnimState:PlayAnimation("ghost_orchid_stage_2_to_3")
                inst.SoundEmitter:PlaySound("dontstarve/common/ghost_spawn")
                
                inst:DoTaskInTime(1, function()
                    inst.stage = STAGE.TWO
                    inst:PushEvent("ghost_orchid_stage_changed", {stage = inst.stage})
                    inst.AnimState:PlayAnimation("ghost_orchid_stage2_idle", true)
                    
                    -- 更新名称
                    if inst.components.named then
                        inst.components.named:SetName(STRINGS.NAMES.GHOST_ORCHID_STAGE2)
                    end
                    
                    inst.sanity_absorbed = 0
                    
                    -- 开始精神值吸收
                    if not inst.sanity_task then
                        inst.sanity_task = inst:DoPeriodicTask(1, function()
                            local x, y, z = inst.Transform:GetWorldPosition()
                            local players = FindPlayersInRange(x, y, z, 4)
                            
                            for _, player in ipairs(players) do
                                if player and player:IsValid() and player.components.sanity and not player.components.health:IsDead() then
                                    player.components.sanity:DoDelta(-1)
                                    inst.sanity_absorbed = (inst.sanity_absorbed or 0) + 1
                                    
                                    if inst.sanity_absorbed >= 3 then
                                        inst.AnimState:PlayAnimation("ghost_orchid_stage_3_to_4")
                                        inst.SoundEmitter:PlaySound("dontstarve/common/ghost_spawn")
                                        
                                        inst:DoTaskInTime(1, function()
                                            inst.stage = STAGE.THREE
                                            inst:PushEvent("ghost_orchid_stage_changed", {stage = inst.stage})
                                            inst.AnimState:PlayAnimation("ghost_orchid_stage3_idle", true)
                                            
                                            -- 更新名称
                                            if inst.components.named then
                                                inst.components.named:SetName(STRINGS.NAMES.GHOST_ORCHID_STAGE3)
                                            end
                                            
                                            inst:AddTag("resurrector")
                                            
                                            if inst.sanity_task then
                                                inst.sanity_task:Cancel()
                                                inst.sanity_task = nil
                                            end
                                        end)
                                        break
                                    end
                                end
                            end
                        end)
                    end
                end)
            end)
        end)

        return true
    end
    return false
end

-- 复活功能
local function OnResurrect(inst, doer)
    if inst.stage == STAGE.THREE and doer and doer:HasTag("playerghost") then
        -- 播放第一个动画
        inst.AnimState:PlayAnimation("ghost_orchid_resurrect_use")
        inst.SoundEmitter:PlaySound("dontstarve/common/ghost_spawn")
        
        -- 在第一个动画结束后立即播放第二个动画
        inst:DoTaskInTime(0.8, function()
            -- 播放复活特效和声音
            local fx = SpawnPrefab("resurrectionstone_fx")
            if fx then
                fx.Transform:SetPosition(doer.Transform:GetWorldPosition())
            end
            
            if doer:HasTag("playerghost") then
                doer.resurrection_source = "ghost_orchid"
                doer:PushEvent("respawnfromghost", { source = inst })
            end
            
            inst.SoundEmitter:PlaySound("dontstarve/common/resurrectionstone_activate")
            
            -- 立即播放第二个动画
            inst.AnimState:PlayAnimation("ghost_orchid_resurrect_finish")
            
            -- 在第二个动画结束后生成种子并移除
            inst:DoTaskInTime(0.8, function()
                local seed = SpawnPrefab("ghost_orchid_seed")
                if seed then
                    seed.Transform:SetPosition(inst.Transform:GetWorldPosition())
                end
                inst:Remove()
            end)
        end)
        
        return true
    end
    return false
end

-- 铲除功能
local function OnDig(inst, worker)
    if worker and worker.components.inventory then
        inst.SoundEmitter:PlaySound("dontstarve/wilson/dig")
        
        if inst.stage_task then
            inst.stage_task:Cancel()
            inst.stage_task = nil
        end
        
        if inst.sanity_task then
            inst.sanity_task:Cancel()
            inst.sanity_task = nil
        end
        
        local seed = SpawnPrefab("ghost_orchid_seed")
        if seed then
            seed.Transform:SetPosition(inst.Transform:GetWorldPosition())
            worker.components.inventory:GiveItem(seed, nil, inst:GetPosition())
        end
        
        inst:Remove()
        return true
    end
    return false
end

local function fn()
    local inst = CreateEntity()

    inst.entity:AddTransform()
    inst.entity:AddAnimState()
    inst.entity:AddNetwork()
    inst.entity:AddSoundEmitter()

    inst.AnimState:SetBank("entity")
    inst.AnimState:SetBuild("ghost_orchid")
    inst.AnimState:PlayAnimation("ghost_orchid_stage1_idle", true)

    -- 设置光照
    inst.entity:AddLight()
    inst.Light:SetFalloff(0.5)
    inst.Light:SetIntensity(0.8)
    inst.Light:SetRadius(1.5)
    inst.Light:SetColour(180/255, 220/255, 255/255)
    inst.Light:Enable(true)

    MakeObstaclePhysics(inst, 0.3)

    inst:AddTag("ghost_orchid")
    inst:AddTag("structure")
    inst:AddTag("plant")

    -- 客户端代码
    if not TheNet:GetIsServer() then
        inst:ListenForEvent("ghost_orchid_stage_changed", function(inst, data)
            if data and data.stage then
                inst.stage = data.stage
            end
        end)
    end

    inst.entity:SetPristine()

    if not TheWorld.ismastersim then
        return inst
    end

    -- 初始化阶段
    inst.stage = STAGE.ONE

    -- 添加名称组件并设置初始名称
    inst:AddComponent("named")
    inst.components.named:SetName(STRINGS.NAMES.GHOST_ORCHID_STAGE1)

    -- 只添加检查组件
    inst:AddComponent("inspectable")

    -- 可喂食组件（阶段1）
    inst:AddComponent("trader")
    inst.components.trader:SetAcceptTest(function(inst, item, giver)
        return item.prefab == "monstermeat" and inst.stage == STAGE.ONE
    end)
    inst.components.trader.onaccept = OnFeedMonsterMeat

    -- 可挖掘组件
    inst:AddComponent("workable")
    inst.components.workable:SetWorkAction(ACTIONS.DIG)
    inst.components.workable:SetWorkLeft(1)
    inst.components.workable:SetOnFinishCallback(OnDig)

    -- 作祟组件
    inst:AddComponent("hauntable")
    inst.components.hauntable.cooldown = 0
    inst.components.hauntable.cooldown_on_successful_haunt = false
    inst.components.hauntable.hauntvalue = TUNING.HAUNT_MEDIUM
    inst.components.hauntable.onhaunt = function(inst, doer)
        return OnResurrect(inst, doer)
    end

    return inst
end

return Prefab("ghost_orchid", fn, assets, prefabs)